/*
escapeSnippetCharacters

Arguments:
	- text (string)

Returns the string with special characters escaped for safe use in a snippet
*/
library.escapeSnippetCharacters = function(text) {
	return text.replace(/([${}`\\])/g, '\\$1');
};

/*
getWordAndRange

Arguments:
	- context (TextActionContext)
	- range (Range or int): the range or index to search at
	- bidirectional (bool): defaults to false; if true searches left and right instead of just left

Returns an object with the following indices:
	- word (string)
	- range (Range)
*/
library.getWordAndRange = function(context, range, bidirectional) {
	// Create our default value for bidirectional
	bidirectional = (typeof bidirectional !== 'boolean' ? false : bidirectional);
	
	// Helper regex for determining if line ends with a tag
	// Includes checks for ASP/PHP/JSP/ColdFusion closing delimiters
	var re_tag = /(<\/?[\w:\-]+[^>]*|\s*(\/|\?|%|-{2,3}))>$/;
	
	// Utility function to check if the line to index ends with a tag
	function endsWithTag(currentIndex) {
		var linestart = context.linestorage.lineStartIndexLessThanIndex(currentIndex);
		var range = new Range(linestart, currentIndex - linestart + 1);
		var text = context.string.substringWithRange(range);
		return re_tag.test(text);
	}
	
	// Setup basic variables
	var cursor, word = '', character = '', inword = false, lastIndex, firstIndex;
	// Determine our start index based on what was passed to us
	if (typeof range === 'number') {
		cursor = range;
	} else {
		cursor = range.location;
	}
	var index = cursor;
	var maxLength = context.string.length;
	
	if (bidirectional) {
		// Make sure the cursor isn't at the end of the document
		if (index !== maxLength) {
			// Check if the cursor is mid-word
			character = context.string.substringWithRange(new Range(index, 1));
			if (re_wordChar.test(character)) {
				inword = true;
				// Parse forward until we hit the end of the word or document
				while (inword) {
					character = context.string.substringWithRange(new Range(index, 1));
					if (re_wordChar.test(character)) {
						word += character;
					} else {
						inword = false;
					}
					index++;
					if (index === maxLength) {
						inword = false;
					}
				}
			} else {
				// The lastIndex logic assumes we've been incrementing as we go, so bump up one to compensate for not going anywhere
				index++;
			}
		}
		// Set our last index for the word
		lastIndex = (index < maxLength ? index - 1 : index);
	} else {
		// We are only parsing to the left, so the final index will be the cursor
		lastIndex = cursor;
	}
	// Reset the index to one less than the cursor
	index = cursor - 1;
	// Only walk left if we aren't at the beginning of the text
	if (index >= 0) {
		// Parse backward to get the word ahead of the cursor
		inword = true;
		while (inword) {
			character = context.string.substringWithRange(new Range(index, 1));
			if (re_wordChar.test(character) && !(character == '>' && endsWithTag(index))) {
				word = character + word;
				index--;
			} else {
				inword = false;
			}
			if (index < 0) {
				inword = false;
			}
		}
	}
	// Since index is left-aligned and we've overcompensated, need to increase +1
	firstIndex = index + 1;
	return {
		"word": word,
		"range": new Range(firstIndex, lastIndex - firstIndex)
	};
};

/*
itemForRange

Arguments:
	- context (TextActionContext)
	- range (Range)

Returns the smallest itemizer item that contains the passed range
*/
var itemForRange = function(context, range) {
	return context.itemizer.smallestItemContainingCharacterRange(range);
};

/*
itemParentForRange

Arguments:

Returns the parent of the item containing the given range
*/
library.itemParentForRange = function(context, range) {
	var item = itemForRange(context, range);
	if (item === null) {
		return null;
	}
	var newRange = item.range;
	// Select the parent if the range is the same
	while(item.parentItem && (newRange.location == range.location && newRange.length == range.length)) {
		item = item.parentItem;
		newRange = item.range;
	}
	return item;
};

/*
selectFromZones

Checks the keys in alternatives to see what matches the active zone,
and returns that item's contents, or defaultSnippet if no match
*/
library.selectFromZones = function(context, range, defaultSnippet, alternatives) {
	var snippet = defaultSnippet;
	// Parse through the alternatives and see if there's anything that matches
	var selectors, zone;
	for (var key in alternatives) {
		selectors = new SXSelector(key);
		if (context.string.length === range.location) {
			zone = context.syntaxTree.rootZone;
		} else {
			zone = context.syntaxTree.zoneAtCharacterIndex(range.location);
		}
		if (selectors.matches(zone)) {
			snippet = alternatives[key];
			break;
		}
	}
	return snippet;
};

/*
zoneAtIndex

Returns the deepest SXZone at the given character index
*/
library.zoneAtIndex = function(context, index) {
	var zone;
	if (context.string.length === index) {
		zone = context.syntaxTree.rootZone;
	} else {
		zone = context.syntaxTree.zoneAtCharacterIndex(index);
	}
	return zone;
};

/*
trim

Trims whitespace from the text

Arguments:
- context (TextActionContext)
- text (string): the text to trim
- parseLines (bool): if true will trim each line
- targetSide (string): 'both' (default), 'start', or 'end'
- respectIndent (bool): if true, indent characters at the start of lines will be left alone
- preserveLinebreaks (bool): if false, linebreaks will be considered whitespace
- trimEmpty (bool): if true will discard even indent whitespace on empty lines
*/
library.trim = function(context, text, parseLines, targetSide, respectIndent, preserveLinebreaks, trimEmpty) {
	// Utility function for trimming the text
	function trimIt(text) {
		// indentRegex is defined in main trim() function, as are numerous boolean values
		// Setup our basic variables
		var match;
		var indentChars = '';
		// Preserve the indent if an indent string is passed in
		if (respectIndent && (targetSide === 'both' || targetSide === 'start')) {
			
			// Check if the text starts with indent characters, and harvest them if so
			match = text.match(indentRegex);
			if (match !== null) {
				indentChars = match[0];
			}
		}
		// Preserve linebreaks at the end if needed
		var linebreak = '';
		match = text.match(/[\n\r]+$/);
		if (preserveLinebreaks && match !== null) {
			linebreak = match[0];
		}
		// Strip the whitespace!
		if (targetSide === 'start') {
			text = text.replace(/^\s+/, '');
		} else if (targetSide === 'end') {
			text = text.replace(/\s+$/, '');
		} else {
			text = text.replace(/^\s+|\s+$/g, '');
		}
		// Put it all together
		if (trimEmpty && text === '') {
			return linebreak;
		} else {
			return indentChars + text + linebreak;
		}
	}
	
	// Create our indentation matching regex
	var indentRegex = new RegExp('^(' + context.textPreferences.tabString + ')+');
	// Construct our final text
	var finalText = '';
	if (parseLines) {
		// Split the lines; this messed-up regex is the only way I can get the linebreaks to accompany their respective lines
		var lines = text.match(/^.*$(?:[\n\r]+)?/gm);
		var count = lines.length;
		var line;
		for (var i = 0; i < count; i++) {
			line = lines[i];
			finalText += trimIt(line);
		}
	} else {
		finalText = trimIt(text);
	}
	
	return finalText;
};

/*
simpleTitleCase

Does a simple title case adjustment to the passed string (capitalizes all words)

Credit: http://stackoverflow.com/questions/196972/convert-string-to-proper-case-with-javascript
*/
library.toSimpleTitleCase = function(text) {
	return text.replace(/\w\S*/g, toTitleCase);
};
var toTitleCase = function(txt) {
	return txt.charAt(0).toUpperCase() + txt.substr(1).toLowerCase();
}

// Encoding Functions and Helpers

var ampersandRegex = /&(?!([a-zA-Z0-9]+|#[0-9]+|#x[0-9a-fA-F]+);)/g;
var validCharsRegex = /[^\u0009\u000A\u000D\u0020-\u007F]/g;

var encodeAmpersands = function(text) {
	return text.replace(ampersandRegex, '&amp;');
};

var encodeNumericAmpersands = function(text) {
	return text.replace(ampersandRegex, '&#38;');
};

var unicodeToNamedEntities = function(text) {
	return encodeAmpersands(text.replace(validCharsRegex, function(c) {
		return '&' + (entityTable[c.charCodeAt(0)] || '#' + c.charCodeAt(0)) + ';';
	}));
};

var unicodeToNumericEntities = function(text) {
	return encodeAmpersands(text.replace(validCharsRegex, function(c) {
		return '&#' + c.charCodeAt(0) + ';';
	}));
};

/* all HTML4 entities as defined here: http://www.w3.org/TR/html4/sgml/entities.html

Source: http://stackoverflow.com/questions/1354064/how-to-convert-characters-to-html-entities-using-plain-javascript

Removed:
	38 : 'amp',
	34 : 'quot',
	39 : 'apos',
	60 : 'lt',
	62 : 'gt',
*/
	
var entityTable = {
	160 : 'nbsp',
	161 : 'iexcl',
	162 : 'cent',
	163 : 'pound',
	164 : 'curren',
	165 : 'yen',
	166 : 'brvbar',
	167 : 'sect',
	168 : 'uml',
	169 : 'copy',
	170 : 'ordf',
	171 : 'laquo',
	172 : 'not',
	173 : 'shy',
	174 : 'reg',
	175 : 'macr',
	176 : 'deg',
	177 : 'plusmn',
	178 : 'sup2',
	179 : 'sup3',
	180 : 'acute',
	181 : 'micro',
	182 : 'para',
	183 : 'middot',
	184 : 'cedil',
	185 : 'sup1',
	186 : 'ordm',
	187 : 'raquo',
	188 : 'frac14',
	189 : 'frac12',
	190 : 'frac34',
	191 : 'iquest',
	192 : 'Agrave',
	193 : 'Aacute',
	194 : 'Acirc',
	195 : 'Atilde',
	196 : 'Auml',
	197 : 'Aring',
	198 : 'AElig',
	199 : 'Ccedil',
	200 : 'Egrave',
	201 : 'Eacute',
	202 : 'Ecirc',
	203 : 'Euml',
	204 : 'Igrave',
	205 : 'Iacute',
	206 : 'Icirc',
	207 : 'Iuml',
	208 : 'ETH',
	209 : 'Ntilde',
	210 : 'Ograve',
	211 : 'Oacute',
	212 : 'Ocirc',
	213 : 'Otilde',
	214 : 'Ouml',
	215 : 'times',
	216 : 'Oslash',
	217 : 'Ugrave',
	218 : 'Uacute',
	219 : 'Ucirc',
	220 : 'Uuml',
	221 : 'Yacute',
	222 : 'THORN',
	223 : 'szlig',
	224 : 'agrave',
	225 : 'aacute',
	226 : 'acirc',
	227 : 'atilde',
	228 : 'auml',
	229 : 'aring',
	230 : 'aelig',
	231 : 'ccedil',
	232 : 'egrave',
	233 : 'eacute',
	234 : 'ecirc',
	235 : 'euml',
	236 : 'igrave',
	237 : 'iacute',
	238 : 'icirc',
	239 : 'iuml',
	240 : 'eth',
	241 : 'ntilde',
	242 : 'ograve',
	243 : 'oacute',
	244 : 'ocirc',
	245 : 'otilde',
	246 : 'ouml',
	247 : 'divide',
	248 : 'oslash',
	249 : 'ugrave',
	250 : 'uacute',
	251 : 'ucirc',
	252 : 'uuml',
	253 : 'yacute',
	254 : 'thorn',
	255 : 'yuml',
	338 : 'OElig',
	339 : 'oelig',
	352 : 'Scaron',
	353 : 'scaron',
	376 : 'Yuml',
	402 : 'fnof',
	710 : 'circ',
	732 : 'tilde',
	913 : 'Alpha',
	914 : 'Beta',
	915 : 'Gamma',
	916 : 'Delta',
	917 : 'Epsilon',
	918 : 'Zeta',
	919 : 'Eta',
	920 : 'Theta',
	921 : 'Iota',
	922 : 'Kappa',
	923 : 'Lambda',
	924 : 'Mu',
	925 : 'Nu',
	926 : 'Xi',
	927 : 'Omicron',
	928 : 'Pi',
	929 : 'Rho',
	931 : 'Sigma',
	932 : 'Tau',
	933 : 'Upsilon',
	934 : 'Phi',
	935 : 'Chi',
	936 : 'Psi',
	937 : 'Omega',
	945 : 'alpha',
	946 : 'beta',
	947 : 'gamma',
	948 : 'delta',
	949 : 'epsilon',
	950 : 'zeta',
	951 : 'eta',
	952 : 'theta',
	953 : 'iota',
	954 : 'kappa',
	955 : 'lambda',
	956 : 'mu',
	957 : 'nu',
	958 : 'xi',
	959 : 'omicron',
	960 : 'pi',
	961 : 'rho',
	962 : 'sigmaf',
	963 : 'sigma',
	964 : 'tau',
	965 : 'upsilon',
	966 : 'phi',
	967 : 'chi',
	968 : 'psi',
	969 : 'omega',
	977 : 'thetasym',
	978 : 'upsih',
	982 : 'piv',
	8194 : 'ensp',
	8195 : 'emsp',
	8201 : 'thinsp',
	8204 : 'zwnj',
	8205 : 'zwj',
	8206 : 'lrm',
	8207 : 'rlm',
	8211 : 'ndash',
	8212 : 'mdash',
	8216 : 'lsquo',
	8217 : 'rsquo',
	8218 : 'sbquo',
	8220 : 'ldquo',
	8221 : 'rdquo',
	8222 : 'bdquo',
	8224 : 'dagger',
	8225 : 'Dagger',
	8226 : 'bull',
	8230 : 'hellip',
	8240 : 'permil',
	8242 : 'prime',
	8243 : 'Prime',
	8249 : 'lsaquo',
	8250 : 'rsaquo',
	8254 : 'oline',
	8260 : 'frasl',
	8364 : 'euro',
	8472 : 'weierp',
	8465 : 'image',
	8476 : 'real',
	8482 : 'trade',
	8501 : 'alefsym',
	8592 : 'larr',
	8593 : 'uarr',
	8594 : 'rarr',
	8595 : 'darr',
	8596 : 'harr',
	8629 : 'crarr',
	8656 : 'lArr',
	8657 : 'uArr',
	8658 : 'rArr',
	8659 : 'dArr',
	8660 : 'hArr',
	8704 : 'forall',
	8706 : 'part',
	8707 : 'exist',
	8709 : 'empty',
	8711 : 'nabla',
	8712 : 'isin',
	8713 : 'notin',
	8715 : 'ni',
	8719 : 'prod',
	8721 : 'sum',
	8722 : 'minus',
	8727 : 'lowast',
	8730 : 'radic',
	8733 : 'prop',
	8734 : 'infin',
	8736 : 'ang',
	8743 : 'and',
	8744 : 'or',
	8745 : 'cap',
	8746 : 'cup',
	8747 : 'int',
	8756 : 'there4',
	8764 : 'sim',
	8773 : 'cong',
	8776 : 'asymp',
	8800 : 'ne',
	8801 : 'equiv',
	8804 : 'le',
	8805 : 'ge',
	8834 : 'sub',
	8835 : 'sup',
	8836 : 'nsub',
	8838 : 'sube',
	8839 : 'supe',
	8853 : 'oplus',
	8855 : 'otimes',
	8869 : 'perp',
	8901 : 'sdot',
	8968 : 'lceil',
	8969 : 'rceil',
	8970 : 'lfloor',
	8971 : 'rfloor',
	9001 : 'lang',
	9002 : 'rang',
	9674 : 'loz',
	9824 : 'spades',
	9827 : 'clubs',
	9829 : 'hearts',
	9830 : 'diams'
};

// Unicode letter definitions borrowed from XRegExp: http://xregexp.com/
var re_wordChar = /^[\u0041-\u005A\u0061-\u007A\u00AA\u00B5\u00BA\u00C0-\u00D6\u00D8-\u00F6\u00F8-\u02C1\u02C6-\u02D1\u02E0-\u02E4\u02EC\u02EE\u0370-\u0374\u0376\u0377\u037A-\u037D\u0386\u0388-\u038A\u038C\u038E-\u03A1\u03A3-\u03F5\u03F7-\u0481\u048A-\u0525\u0531-\u0556\u0559\u0561-\u0587\u05D0-\u05EA\u05F0-\u05F2\u0621-\u064A\u066E\u066F\u0671-\u06D3\u06D5\u06E5\u06E6\u06EE\u06EF\u06FA-\u06FC\u06FF\u0710\u0712-\u072F\u074D-\u07A5\u07B1\u07CA-\u07EA\u07F4\u07F5\u07FA\u0800-\u0815\u081A\u0824\u0828\u0904-\u0939\u093D\u0950\u0958-\u0961\u0971\u0972\u0979-\u097F\u0985-\u098C\u098F\u0990\u0993-\u09A8\u09AA-\u09B0\u09B2\u09B6-\u09B9\u09BD\u09CE\u09DC\u09DD\u09DF-\u09E1\u09F0\u09F1\u0A05-\u0A0A\u0A0F\u0A10\u0A13-\u0A28\u0A2A-\u0A30\u0A32\u0A33\u0A35\u0A36\u0A38\u0A39\u0A59-\u0A5C\u0A5E\u0A72-\u0A74\u0A85-\u0A8D\u0A8F-\u0A91\u0A93-\u0AA8\u0AAA-\u0AB0\u0AB2\u0AB3\u0AB5-\u0AB9\u0ABD\u0AD0\u0AE0\u0AE1\u0B05-\u0B0C\u0B0F\u0B10\u0B13-\u0B28\u0B2A-\u0B30\u0B32\u0B33\u0B35-\u0B39\u0B3D\u0B5C\u0B5D\u0B5F-\u0B61\u0B71\u0B83\u0B85-\u0B8A\u0B8E-\u0B90\u0B92-\u0B95\u0B99\u0B9A\u0B9C\u0B9E\u0B9F\u0BA3\u0BA4\u0BA8-\u0BAA\u0BAE-\u0BB9\u0BD0\u0C05-\u0C0C\u0C0E-\u0C10\u0C12-\u0C28\u0C2A-\u0C33\u0C35-\u0C39\u0C3D\u0C58\u0C59\u0C60\u0C61\u0C85-\u0C8C\u0C8E-\u0C90\u0C92-\u0CA8\u0CAA-\u0CB3\u0CB5-\u0CB9\u0CBD\u0CDE\u0CE0\u0CE1\u0D05-\u0D0C\u0D0E-\u0D10\u0D12-\u0D28\u0D2A-\u0D39\u0D3D\u0D60\u0D61\u0D7A-\u0D7F\u0D85-\u0D96\u0D9A-\u0DB1\u0DB3-\u0DBB\u0DBD\u0DC0-\u0DC6\u0E01-\u0E30\u0E32\u0E33\u0E40-\u0E46\u0E81\u0E82\u0E84\u0E87\u0E88\u0E8A\u0E8D\u0E94-\u0E97\u0E99-\u0E9F\u0EA1-\u0EA3\u0EA5\u0EA7\u0EAA\u0EAB\u0EAD-\u0EB0\u0EB2\u0EB3\u0EBD\u0EC0-\u0EC4\u0EC6\u0EDC\u0EDD\u0F00\u0F40-\u0F47\u0F49-\u0F6C\u0F88-\u0F8B\u1000-\u102A\u103F\u1050-\u1055\u105A-\u105D\u1061\u1065\u1066\u106E-\u1070\u1075-\u1081\u108E\u10A0-\u10C5\u10D0-\u10FA\u10FC\u1100-\u1248\u124A-\u124D\u1250-\u1256\u1258\u125A-\u125D\u1260-\u1288\u128A-\u128D\u1290-\u12B0\u12B2-\u12B5\u12B8-\u12BE\u12C0\u12C2-\u12C5\u12C8-\u12D6\u12D8-\u1310\u1312-\u1315\u1318-\u135A\u1380-\u138F\u13A0-\u13F4\u1401-\u166C\u166F-\u167F\u1681-\u169A\u16A0-\u16EA\u1700-\u170C\u170E-\u1711\u1720-\u1731\u1740-\u1751\u1760-\u176C\u176E-\u1770\u1780-\u17B3\u17D7\u17DC\u1820-\u1877\u1880-\u18A8\u18AA\u18B0-\u18F5\u1900-\u191C\u1950-\u196D\u1970-\u1974\u1980-\u19AB\u19C1-\u19C7\u1A00-\u1A16\u1A20-\u1A54\u1AA7\u1B05-\u1B33\u1B45-\u1B4B\u1B83-\u1BA0\u1BAE\u1BAF\u1C00-\u1C23\u1C4D-\u1C4F\u1C5A-\u1C7D\u1CE9-\u1CEC\u1CEE-\u1CF1\u1D00-\u1DBF\u1E00-\u1F15\u1F18-\u1F1D\u1F20-\u1F45\u1F48-\u1F4D\u1F50-\u1F57\u1F59\u1F5B\u1F5D\u1F5F-\u1F7D\u1F80-\u1FB4\u1FB6-\u1FBC\u1FBE\u1FC2-\u1FC4\u1FC6-\u1FCC\u1FD0-\u1FD3\u1FD6-\u1FDB\u1FE0-\u1FEC\u1FF2-\u1FF4\u1FF6-\u1FFC\u2071\u207F\u2090-\u2094\u2102\u2107\u210A-\u2113\u2115\u2119-\u211D\u2124\u2126\u2128\u212A-\u212D\u212F-\u2139\u213C-\u213F\u2145-\u2149\u214E\u2183\u2184\u2C00-\u2C2E\u2C30-\u2C5E\u2C60-\u2CE4\u2CEB-\u2CEE\u2D00-\u2D25\u2D30-\u2D65\u2D6F\u2D80-\u2D96\u2DA0-\u2DA6\u2DA8-\u2DAE\u2DB0-\u2DB6\u2DB8-\u2DBE\u2DC0-\u2DC6\u2DC8-\u2DCE\u2DD0-\u2DD6\u2DD8-\u2DDE\u2E2F\u3005\u3006\u3031-\u3035\u303B\u303C\u3041-\u3096\u309D-\u309F\u30A1-\u30FA\u30FC-\u30FF\u3105-\u312D\u3131-\u318E\u31A0-\u31B7\u31F0-\u31FF\u3400-\u4DB5\u4E00-\u9FCB\uA000-\uA48C\uA4D0-\uA4FD\uA500-\uA60C\uA610-\uA61F\uA62A\uA62B\uA640-\uA65F\uA662-\uA66E\uA67F-\uA697\uA6A0-\uA6E5\uA717-\uA71F\uA722-\uA788\uA78B\uA78C\uA7FB-\uA801\uA803-\uA805\uA807-\uA80A\uA80C-\uA822\uA840-\uA873\uA882-\uA8B3\uA8F2-\uA8F7\uA8FB\uA90A-\uA925\uA930-\uA946\uA960-\uA97C\uA984-\uA9B2\uA9CF\uAA00-\uAA28\uAA40-\uAA42\uAA44-\uAA4B\uAA60-\uAA76\uAA7A\uAA80-\uAAAF\uAAB1\uAAB5\uAAB6\uAAB9-\uAABD\uAAC0\uAAC2\uAADB-\uAADD\uABC0-\uABE2\uAC00-\uD7A3\uD7B0-\uD7C6\uD7CB-\uD7FB\uF900-\uFA2D\uFA30-\uFA6D\uFA70-\uFAD9\uFB00-\uFB06\uFB13-\uFB17\uFB1D\uFB1F-\uFB28\uFB2A-\uFB36\uFB38-\uFB3C\uFB3E\uFB40\uFB41\uFB43\uFB44\uFB46-\uFBB1\uFBD3-\uFD3D\uFD50-\uFD8F\uFD92-\uFDC7\uFDF0-\uFDFB\uFE70-\uFE74\uFE76-\uFEFC\uFF21-\uFF3A\uFF41-\uFF5A\uFF66-\uFFBE\uFFC2-\uFFC7\uFFCA-\uFFCF\uFFD2-\uFFD7\uFFDA-\uFFDC-]$/;

// Expose functions
library.itemForRange = itemForRange;
library.encodeAmpersands = encodeAmpersands;
library.encodeNumericAmpersands = encodeNumericAmpersands;
library.unicodeToNamedEntities = unicodeToNamedEntities;
library.unicodeToNumericEntities = unicodeToNumericEntities;

library.ampersandRegex = ampersandRegex;
library.validCharsRegex = validCharsRegex;
